;;; mule-msw.el --- Mule support stuff for MS Windows
;; Copyright (C) 2000 IKEYAMA Tomonori

;; This file is part of XEmacs.
;;
;; XEmacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; XEmacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Synched up with: Not in FSF.

;;; Commentary:

;;; Code:

;; Charsets to Windows Codepage mapping.
(let ((l '((ascii . 1252)
	   (latin-iso8859-2 . 1250)
	   (cyrillic-iso8859-5 . 1251)
	   (latin-iso8859-1 . 1252)
	   (greek-iso8859-7 . 1253) 
	   (latin-iso8859-9 . 1254)
	   (hebrew-iso8859-8 . 1255)
	   ;; (arabic-iso8859-6 . 1256)
	   (latin-iso8859-4 . 1257)
	   ;; (vietnamese-viscii-lower . 1258)
	   ;; (vietnamese-viscii-upper . 1258)
	   ;; (thai-tis620 . 874)
	   (latin-jisx0201 . 932)
	   (katakana-jisx0201 . 932)
	   (japanese-jisx0208-1978 . 932)
	   (japanese-jisx0208 . 932)
	   ;; (japanese-jisx0212 . 932)
	   (chinese-gb2312 . 936)
	   (korean-ksc5601 . 949)
	   (chinese-big5-1 . 950)
	   (chinese-big5-2 . 950))))
  (while l
    (let ((charset (car (car l)))
	  (codepage (cdr (car l))))
    (mswindows-set-charset-codepage charset codepage)
    (setq l (cdr l)))))

;; Charset to font encoder mapping.

;; Japanese JISX0208
(let ((ccl-program
       (ccl-compile
	'(0
	  ((r1 = (r1 en-sjis r2))
	   (r2 = r7))))))
  (mswindows-set-charset-ccl-program 'japanese-jisx0208 ccl-program)
  (mswindows-set-charset-ccl-program 'japanese-jisx0208-1978 ccl-program))
      
;; Chinese GB2312 and Korean KSC5601
(let ((ccl-program
       (ccl-compile
	'(0
	  (r1 |= ?\x80)
	  (r2 |= ?\x80)))))
  (mswindows-set-charset-ccl-program 'chinese-gb2312 ccl-program)
  (mswindows-set-charset-ccl-program 'korean-ksc5601 ccl-program))

;; Chinese BIG5
(mswindows-set-charset-ccl-program 'chinese-big5-1 ccl-encode-big5-font)
(mswindows-set-charset-ccl-program 'chinese-big5-2 ccl-encode-big5-font)

;; Here are Windows Codepage decoding tables from the codepage.el in
;; FSF Emacs. The codepage.el doesn't work on XEmacs because our CCL
;; dosen't have 'translate-character command. This is available only
;; for font encoder at the moment.

;; start importing from codepage.el

;; Support for the Windows 12xx series of codepages that MS has
;; butchered from the ISO-8859 specs. This does not add support for
;; the extended characters that MS has added in the 128 - 159 coding
;; range, only translates those characters that can be expressed in
;; the corresponding iso-8859 codepage.

;; Codepage Mapping:
;;
;; Windows-1250: ISO-8859-2 (Central Europe) - differs in some positions
;; Windows-1251: ISO-8859-5 (Cyrillic)       - differs wildly
;; Windows-1252: ISO-8859-1 (West Europe)    - exact match
;; Windows-1253: ISO-8859-7 (Greek)          - differs in some positions
;; Windows-1254: ISO-8859-9 (Turkish)        - exact match
;; Windows-1255: ISO-8859-8 (Hebrew)         - exact match
;; Windows-1256: ISO-8859-6 (Arabic)         - half match
;; Windows-1257: ISO-8859-4 (Baltic)         - differs, future Latin-7
;; Windows-1258: VISCII (Vietnamese)         - Completely different

(defvar cp1250-decode-table
  [
   160 165 162 163 164 188 140 167 168 138 170 141 143 173 142 175
   176 185 178 179 180 190 156 161 184 154 186 157 159 189 158 191
   192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207
   208 209 210 211 212 213 214 215 216 217 218 219 220 221 222 223
   224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239
   240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 255 ]
  "ISO-8859-2 to Windows-1250 (Central Europe) codepage decoding table")
(setplist 'cp1250-decode-table
	  '(charset latin-iso8859-2 language "Latin-2" offset 160))

(defvar cp1251-decode-table
  [
   160 168 128 129 170 189 178 175 163 138 140 142 141 173 161 143
   192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207
   208 209 210 211 212 213 214 215 216 217 218 219 220 221 222 223
   224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239
   240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 255
   185 184 144 131 186 190 179 191 188 154 156 158 157 167 162 159 ]
  "ISO-8859-5 to Windows-1251 (Cyrillic) codepage decoding table")
(setplist 'cp1251-decode-table
	  '(charset cyrillic-iso8859-5 language "Cyrillic-ISO" offset 160))

;; cp1253 is missing nbsp so we cannot quite translate perfectly. It
;; also has two micro/mu characters which would require more complex
;; processing to accomodate.
(defvar cp1253-decode-table
  [
   nil 145 146 163 nil nil 166 167 168 169 nil 171 172 173 nil 151
   176 177 178 179 180 161 162 183 184 185 186 187 188 189 190 191
   192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207
   208 209 nil 211 212 213 214 215 216 217 218 219 220 221 222 223
   224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239
   240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 nil ]
  "ISO-8859-7 to Windows-1253 (Greek) codepage decoding table")
(setplist 'cp1253-decode-table
	  '(charset greek-iso8859-7 language "Greek" offset 160))

;; Since Latin-7 is not yet official, and Emacs does not support it,
;; provide translation between Windows-1257 and Latin-4 the best we
;; can.
(defvar cp1257-decode-table
  [
   160 192 nil 170 164 nil 207 167 nil 208 199 204 nil 173 222 nil
   176 224 nil 186 nil nil 239 nil nil 240 231 236 nil nil 254 nil
   194 nil nil nil 196 197 175 193 200 201 198 nil 203 nil nil 206
   nil 210 212 205 nil 213 214 215 168 216 nil nil 220 nil 219 223
   226 nil nil nil 228 229 191 225 232 233 230 nil 235 nil nil 238
   nil 242 244 237 nil 245 246 247 184 248 nil nil 252 nil 251 nil ]
  "ISO-8859-4 to Windows-1257 (Baltic) codepage decoding table")
(setplist 'cp1257-decode-table
	  '(charset latin-iso8859-4 language "Latin-4" offset 160))

;; end of import from codepage.el

;; Make font encoders from the decoding tables.

;; Translating all of mappings to CCL 'if' statements. This way is
;; very nasty. Is there better way to make a font encoder for
;; Windows-125x codepages ?
(let ((l '(cp1250-decode-table 
	   cp1251-decode-table
	   cp1253-decode-table
	   cp1257-decode-table)))
  (while l
    (let ((decoder (car l))
	  (i 0)
	  offset charset table len
	  font-encoder ccl-program)
      (setq offset (get decoder 'offset)
	    charset (get decoder 'charset)
	    table (symbol-value decoder)
	    len (length table))
      (while (< i len)
	(let ((from (+ i offset))
	      (to (aref table i)))
	  (if (not to)
	      ;; map to '~'
	      (setq to ?~))
	  (if (not (eq from to))
	      (setq font-encoder (cons `(if (r1 == ,from) (r1 = ,to)) 
				       font-encoder)))
	  (setq i (1+ i))))
      (setq ccl-program (ccl-compile `(0 ,font-encoder)))
      (mswindows-set-charset-ccl-program charset ccl-program)
      (setq l (cdr l)))))

;; Codepage to language and coding system mappings. 
;; To complete this list, we need more coding systems adapted for
;; Windows codepages.
(defvar mswindows-codepage-info-alist
  '(
    ;; (1250 (language . "Latin-2")
    ;;  (coding-system . windows-1250))
    (1251 (language . "Cyrillic-Win")
	  (coding-system . windows-1251))
    (1252 (language . "Latin-1")
	  (coding-system . iso-8859-1))
    ;; (1253 (language . "Greek")
    ;;	  (coding-system . windows-1253))
    (1254 (language . "Latin-5")
	  (coding-system . iso-8859-9))
    (1255 (language . "Hebrew")
	  (coding-system . iso-8859-8))
    ;; (1256 (language . "Arabic")
    ;;	  (coding-system . windows-1256))
    ;; (1257 (language . "Baltic")
    ;;	  (coding-system . windows-1257))
    ;; (1258 (language . "Vietnamese")
    ;;	  (coding-system . windows-1258))
    ;; (874 (language . "Thai")
    ;;	  (coding-system . windows-874))
    (932 (language . "Japanese")
	 (coding-system . shift_jis))
    ;; (936 (language . "Chinese-GB")
    ;;	 (coding-system . gb2312))
    ;; (949 (language . "Korean")
    ;;	 (coding-system . ksc5601))
    (950 (language . "Chinese-BIG5")
	 (coding-system . big5))))

(defun mswindows-set-codepage-envirionment (codepage)
  "Set language environment and native coding system for the CODEPAGE."
  (let ((slot (cdr (assq codepage mswindows-codepage-info-alist)))
	language coding-system)
    ;; language
    (setq language (cdr (assq 'language slot)))
    (if language
	(set-language-environment language))
    ;; coding system
    (setq coding-system (cdr (assq 'coding-system slot)))
    (when coding-system
      (setq coding-system (coding-system-eol-crlf coding-system))
      (prefer-coding-system coding-system)
      ;; Snatch 'file-name from file-name-coding-system for setting
      ;; the coding system depend on the current codepage.
      (dontusethis-set-symbol-value-handler 'file-name-coding-system
					    'set-value nil)
      (define-coding-system-alias 'file-name coding-system))))

(defun mswindows-setup-for-system-codepage ()
  "Setup environment fot the current system's codepage."
  (mswindows-set-codepage-envirionment (mswindows-system-codepage)))

;; Should be called after init-mule.
(add-hook 'before-init-hook 'mswindows-setup-for-system-codepage t)
